/*********************************************************
  PB207 Sequencer Total Recall
  ********************************************************
  Software for the Paper Bit 207 Sequencer Totall Recall

  TRIGGER MODE:
  1. Normal Mode (Ex-Trigger = Off): 
       generates a sequence
  2. Triggered Step (Ex-Trigger = On and Speep Poti = 0):
       each step is triggered by an external signal
  3. Triggered Sequence (Ex-Trigger = On and Speep Poti > 0):
       each external trigger signale will generate a sequence

  STEP MODE:
  1. Multi Step Mode:
       generates a sequence with 2 to 8 steps
  2. Single Step Mode or Step Selection:
       one step is selected, depending on the possition of the Speed Poti
  
  Wolfgang Spahn                          dernulleffekt.de
  8 April 2020
  ********************************************************/

#include <EEPROM.h>

// pin assignmet
int StepPin[8] =  {19, 13, 12, 9, 8, 1, 2, 0};  // pins for each step, buttons & LEDs (19 = A5)

int CVPotiPin = A1;
int StepsPotiPin = A4;
int SpeedPotiPin = A3;
//int MultiplexPin = A2;

int CV1OutPin = 5;
int CV2OutPin = 6;
int TriggerOutPin = 7;
int TriggerInPin = 11;

int TriggerSwitchPin = 10;
int ModeSwitchPin = 3;
int CVSwitchPin = 4;
int CallButtonPin = 14; // A0
int StoreButtonPin = 16;   //A2

// variables assignment
int CV1_Values[8] = {0, 0, 0, 0, 0, 0, 0, 0};                   // voltage output for CV 1
int CV2_Values[8] = {0, 0, 0, 0, 0, 0, 0, 0};                   // voltage output for CV 2
int B_Values[8] = {127, 127, 127, 127, 127, 127, 127, 127};     // length of each beat
int TriggerLength[8] = {10, 10, 10, 10, 10, 10, 10, 10};        // length of the trigger signal

int CV1_Value;
int CV2_Value;
int CV_Value;
int Trigger_Value;
int Groove_Value;

int MultiplexValue = 0;
int CallButton_State = 0;
int StoreButton_State = 0;
int ActiveStep = 0;
int WhichButtonPressed = -1;
int ActivationCallCheck = 1;
int ActivationStoreCheck = 1;
int ButtonStop = 0;

int CV1_Out = 0;
int CV2_Out = 0;
int TriggerOut = 0;
int TriggerTime = 2000;
int TriggerCheck = 0;

int StepsPoti = 1023;
int Steps = 8;
int StepButton = 0;
int StepNumber = 0;
int old_StepNumber = 0;
int CVIn = 0;
unsigned long DelayValue = 0;

int ModeSwitch = LOW;
int CVSwitch = LOW;
int SpeedAmp = 0;
int BeatButton = 0;

int TriggerSwitch = LOW;
int TriggerIn = LOW;
int TriggerInCheck = LOW;

unsigned long previousMillis = 0;
unsigned long currentMillis = 0;
int LEDState = HIGH;

int verbose = 0;  // in verbose mode Step 6 is allways on!



void setup() {
  // sets the speed on pin 5 and 6 (just for a Atmega328 chip):
  TCCR0B = TCCR0B & 0b11111000 | 0x01;

  pinMode(CV1OutPin, OUTPUT);
  analogWrite(CV1OutPin, 0);
  pinMode(CV2OutPin, OUTPUT);
  analogWrite(CV2OutPin, 0);
  pinMode(TriggerOutPin, OUTPUT);
  digitalWrite(TriggerOutPin, LOW);

  pinMode(TriggerInPin, INPUT_PULLUP);
  pinMode(TriggerSwitchPin, INPUT_PULLUP);
  pinMode(ModeSwitchPin, INPUT_PULLUP);
  pinMode(CVSwitchPin, INPUT_PULLUP);
  pinMode(CallButtonPin, INPUT_PULLUP);
  pinMode(StoreButtonPin, INPUT_PULLUP);

  for (int i = 0; i < 8; i++) {
    pinMode(StepPin[i], OUTPUT);
  }
  TestLEDs();
  TriggerTest();
  for (int i = 0; i < 8; i++) {
    pinMode(StepPin[i], INPUT);
  }
  if (verbose > 0) {
    Serial.begin(9600);
    Serial.println("verbose mode is on");
  }

  previousMillis = millis();
}



void loop() {
  // check the trigger switch
  TriggerSwitch = digitalRead(TriggerSwitchPin);
  if (TriggerSwitch == LOW) {
    SelfTrigger();
  }
  if (TriggerSwitch == HIGH) {
    if (analogRead(SpeedPotiPin) > 0) {
      SequenceTrigger();
    }
    else {
      ExTrigger();
    }
  }
}


// for external trigger
void ExTrigger() {
  for (int i = 0; i < Steps; i++) {
    LEDState = LOW;
    pinMode(StepPin[i], OUTPUT);
    digitalWrite(StepPin[i], LEDState);
    analogWrite(CV1OutPin, CV1_Values[i]);
    analogWrite(CV2OutPin, CV2_Values[i]);
    if (TriggerLength[i] > 0) digitalWrite(TriggerOutPin, HIGH);
    TriggerCheck = 0;
    while (LEDState == LOW) {
      ActiveStep = i;
      CheckProgramSwitch();
      if (CallButton_State == 1) CallPattern();
      if (StoreButton_State == 1) StorePattern();
      if (ButtonStop == 0) ButtonCheck();

      StepsPoti = analogRead(StepsPotiPin);
      Steps = map(StepsPoti, 0, 1023, 1, 8);

      TriggerTime = 200 * TriggerLength[i];
      currentMillis = millis();

      if ((currentMillis - previousMillis >= TriggerTime) && (TriggerCheck == 0)) {
        digitalWrite(TriggerOutPin, LOW);
        TriggerCheck = 1;
      }
      TriggerIn = digitalRead(TriggerInPin);

      if (TriggerIn == LOW) TriggerInCheck = 0;

      if ((TriggerIn == HIGH)  && (TriggerInCheck == 0)) {
        previousMillis = currentMillis;
        TriggerInCheck = 1;
        LEDState = HIGH;
        digitalWrite(StepPin[i], LEDState);
        pinMode(StepPin[i], INPUT);
        digitalWrite(TriggerOutPin, LOW);
      }
      TriggerSwitch = digitalRead(TriggerSwitchPin);
      if (TriggerSwitch == LOW) {
        LEDState = HIGH;
        digitalWrite(StepPin[i], LEDState);
        pinMode(StepPin[i], INPUT);
        digitalWrite(TriggerOutPin, LOW);
      }
    }
  }
}

void SequenceTrigger() {
  TriggerIn = digitalRead(TriggerInPin);

  if (TriggerIn == LOW) {
    ActiveStep = 8;
    CheckProgramSwitch();
    if (CallButton_State == 1) CallPattern();
    if (StoreButton_State == 1) StorePattern();
    if (ButtonStop == 0) ButtonCheck();

    TriggerInCheck = 0;
  }

  if ((TriggerIn == HIGH)  && (TriggerInCheck == 0)) {
    for (int i = 0; i < Steps; i++) {
      LEDState = LOW;
      pinMode(StepPin[i], OUTPUT);
      digitalWrite(StepPin[i], LEDState);
      analogWrite(CV1OutPin, CV1_Values[i]);
      analogWrite(CV2OutPin, CV2_Values[i]);
      if (TriggerLength[i] > 0) digitalWrite(TriggerOutPin, HIGH);
      TriggerCheck = 0;
      // for the first step
      if (i == 0) {
        previousMillis = millis();
        while (LEDState == LOW) {
          ActiveStep = i;
          CheckProgramSwitch();
          if (CallButton_State == 1) CallPattern();
          if (StoreButton_State == 1) StorePattern();
          if (ButtonStop == 0) ButtonCheck();

          StepsPoti = analogRead(StepsPotiPin);
          Steps = map(StepsPoti, 0, 1023, 1, 8);

          DelayValue = analogRead(SpeedPotiPin);
          DelayValue = map(DelayValue, 1023, 0, 1, 1500);

          DelayValue = DelayValue * B_Values[i];

          // defines the lengh of the trigger signal
          TriggerTime = 200 * TriggerLength[i];
          
          currentMillis = millis();
          if ((currentMillis - previousMillis >= TriggerTime) && (TriggerCheck == 0)) {
            digitalWrite(TriggerOutPin, LOW);
            TriggerCheck = 1;
          }
          if (currentMillis - previousMillis >= DelayValue) {
            previousMillis = currentMillis;
            LEDState = HIGH;
            digitalWrite(StepPin[i], LEDState);
            pinMode(StepPin[i], INPUT);
            digitalWrite(TriggerOutPin, LOW);
          }
        }
      }
      // for all the other steps
      else {
        while (LEDState == LOW) {
          ActiveStep = i;
          CheckProgramSwitch();
          if (CallButton_State == 1) CallPattern();
          if (StoreButton_State == 1) StorePattern();
          if (ButtonStop == 0) ButtonCheck();

          StepsPoti = analogRead(StepsPotiPin);
          Steps = map(StepsPoti, 0, 1023, 1, 8);

          DelayValue = analogRead(SpeedPotiPin);
          DelayValue = map(DelayValue, 1023, 0, 1, 1500);

          DelayValue = DelayValue * B_Values[i];

          // defines the lengh of the trigger signal
          TriggerTime = 200 * TriggerLength[i];

          currentMillis = millis();
          if ((currentMillis - previousMillis >= TriggerTime) && (TriggerCheck == 0)) {
            digitalWrite(TriggerOutPin, LOW);
            TriggerCheck = 1;
          }
          if (currentMillis - previousMillis >= DelayValue) {
            previousMillis = currentMillis;
            LEDState = HIGH;
            digitalWrite(StepPin[i], LEDState);
            pinMode(StepPin[i], INPUT);
            digitalWrite(TriggerOutPin, LOW);
          }
        }
      }
    }
    analogWrite(CV1OutPin, 3);
    analogWrite(CV2OutPin, 3);
    TriggerInCheck = 1;
  }
}

// for internal trigger
void SelfTrigger() {
  // normal step mode
  if (Steps > 1) {
    for (int i = 0; i < Steps; i++) {
      LEDState = LOW;
      pinMode(StepPin[i], OUTPUT);
      digitalWrite(StepPin[i], LEDState);
      analogWrite(CV1OutPin, CV1_Values[i]);
      analogWrite(CV2OutPin, CV2_Values[i]);
      if (TriggerLength[i] > 0) digitalWrite(TriggerOutPin, HIGH);
      TriggerCheck = 0;
      while (LEDState == LOW) {
        ActiveStep = i;
        CheckProgramSwitch();
        if (CallButton_State == 1) CallPattern();
        if (StoreButton_State == 1) StorePattern();
        if (ButtonStop == 0) ButtonCheck();

        StepsPoti = analogRead(StepsPotiPin);
        Steps = map(StepsPoti, 0, 1023, 1, 8);

        DelayValue = analogRead(SpeedPotiPin);
        DelayValue = map(DelayValue, 1023, 0, 1, 1500);

        DelayValue = DelayValue * B_Values[i];

        // defines the lengh of the trigger signal
        TriggerTime = 200 * TriggerLength[i];

        currentMillis = millis();

        if ((currentMillis - previousMillis >= TriggerTime) && (TriggerCheck == 0)) {
          digitalWrite(TriggerOutPin, LOW);
          TriggerCheck = 1;
        }
        if (currentMillis - previousMillis >= DelayValue) {
          previousMillis = currentMillis;
          LEDState = HIGH;
          digitalWrite(StepPin[i], LEDState);
          pinMode(StepPin[i], INPUT);
          digitalWrite(TriggerOutPin, LOW);
        }
      }
    }
  }

  // single step mode
  if (Steps == 1) {

    StepNumber = analogRead(SpeedPotiPin);

    if (StepNumber < 10) StepNumber = 0;
    StepNumber = map(StepNumber, 1023, 0, 7, 0);
    LEDState = LOW;
    pinMode(StepPin[StepNumber], OUTPUT);
    digitalWrite(StepPin[StepNumber], LEDState);
    analogWrite(CV1OutPin, CV1_Values[StepNumber]);
    analogWrite(CV2OutPin, CV2_Values[StepNumber]);

    ActiveStep = StepNumber;
    CheckProgramSwitch();
    if (CallButton_State == 1) CallPattern();
    if (StoreButton_State == 1) StorePattern();
    if (ButtonStop == 0) ButtonCheck();

    StepsPoti = analogRead(StepsPotiPin);
    Steps = map(StepsPoti, 0, 1023, 1, 8);

    LEDState = HIGH;
    digitalWrite(StepPin[StepNumber], LEDState);
    pinMode(StepPin[StepNumber], INPUT);

    // Trigger
    currentMillis = millis();
    if (StepNumber == old_StepNumber) {
      if ((currentMillis - previousMillis >= TriggerTime) && (TriggerCheck == 0)) {
        digitalWrite(TriggerOutPin, LOW);
        TriggerCheck = 1;
      }
    }
    else {
      digitalWrite(TriggerOutPin, HIGH);
      previousMillis = currentMillis;
      TriggerTime = 200 * TriggerLength[StepNumber];
      TriggerCheck = 0;
      old_StepNumber = StepNumber;
    }
  }
}



//**** for development ********************************************************


// trigger test
void ExTriggerTest() {
  int test = digitalRead(TriggerInPin);
  digitalWrite(TriggerOutPin, test);
}

// testing all LEDs
void TestLEDs() {
  for (int i = 0; i < 8; i++) {
    digitalWrite(StepPin[i], LOW);
  }
  delay(30000);
  for (int i = 0; i < 8; i++) {
    digitalWrite(StepPin[i], HIGH);
    delay(10000);
  }
}

// testing the trigger
void TriggerTest() {
  digitalWrite(TriggerOutPin, HIGH);
  delay(10000);
  digitalWrite(TriggerOutPin, LOW);
}
